// eprime mode: 
//  - LED A-E are eprime on-board lamp 1-5
//  - LED F-H are eprime external lamp 6-8
//  - button A-E are eprime on-board key 1-5
//  - NOT: button F-H are eprime external key 9-11
//  - button F is eprime external key #6 which is also used for the sound key
//  - button G is eprime external key #7 which is for instance for the light sensor high
//  - button H is eprime external key #8
//  - both voice key and sound key trigger voice key/external key #6
//  - 800Hz with 19200 bit/s appears to be the standard setting in ePrime
// extended mode:
// M: set marker 
// P: send pulse
// X: set pulse duration in ms (for P)
// 
///

//#include <digitalWrite.h> // c++ compliant version of digitalWrite.h
#include <EEPROM.h>
#include <SPI.h>
#include <SoftwareSerial.h>
#define rxPin 27
#define txPin 0

// version must be set in Makefile by git
#ifndef VERSION
#define VERSION "unknown"
#endif

#define CPU_RESTART_ADDR (uint32_t *)0xE000ED0C
#define CPU_RESTART_VAL 0x5FA0004
#define CPU_RESTART (*CPU_RESTART_ADDR = CPU_RESTART_VAL);

/* arduino.h:

#define lowByte(w) ((uint8_t) ((w) & 0xff))
#define highByte(w) ((uint8_t) ((w) >> 8))

#define bitRead(value, bit) (((value) >> (bit)) & 0x01)
#define bitSet(value, bit) ((value) |= (1UL << (bit)))
#define bitClear(value, bit) ((value) &= ~(1UL << (bit)))
#define bitWrite(value, bit, bitvalue) (bitvalue ? bitSet(value, bit) : bitClear(value, bit))
*/

const int addressMode = 0; // eprom address where mode is stored
// available modes
const byte modeNone        = 0;
const byte modeBitsi       = 1; // H + A
const byte modeBitsiExtend = 2; // H + B
const byte modePersonal    = 3; // H + C, undocumented mode
const byte modeEprime      = 5; // H + E
const byte modeInitial     = modeNone; // if not modeNone, then change mode at startup

// pin numbers (hardware specific)
const byte pinDataOut     = 11; // MOSI 
const byte pinDataIn      = 12; // MISO 
const byte pinSpiClock    = 13; // Clock 
const byte pinAnalogOut1  = A21;
const byte pinAnalogOut2  = A22;
const byte pinLed         = 3;
const byte pinAnalogIn1   = A14;
const byte pinAnalogIn2   = A15;
const byte pinAnalogIn3   = A16;
const byte pinAnalogIn4   = A15;
//const byte pinInput[8]  = {6, 7, 8, 9, 16, 17, 18, 19}; //duemilanove/uno specific digital input pins for buttons
const byte pinInput[8]    = {15, 16, 17, 22, 23, 20, 21, 14}; //teensy 3.6 specific digital input pins for buttons
const byte pinOutput[8]   = {24, 25, 26, 27, 28, 29, 30, 31}; //teensy 3.6 specific digital input pins for buttons
//const byte pinInput[8]  = { 6, 7, 8, 9, 20, 21, 22, 23}; //leonardo
const byte pinSpeaker     = 5;

// analog channel for sound key and voice key
const byte channelSound   = A20;
const byte channelVoice   = A16;

const byte marginVoice         = 25;  // margin for detection of voice (higher margin = fewer false positives)
const byte marginSound         = 50;  // margin for detection of sound (higher margin = fewer false positives)
const byte timeDebounce        = 2;   // (ms) dead time of a button after button press or release in ms
const unsigned frametimeEprime = 1200;// (ms) 658; // frame time in us for eprime streaming mode
// note that eprime srbox sends about 760 bytes/s when set to 800 bytes/s

// long is 32 bit
unsigned long t0;                     // (ms) time at start of loop (loops in 2^32 ms = 50 days)
unsigned long t0u;                    // (us) time at end of loop in eprime mode (loops in 72 minutes)
unsigned long tPulse = 100;           // (ms) duration of pulse in bitsi extend mode
unsigned long tPulseEnd = 0xffffffff; // (ms) end time of pulse in bitsi extend mode

// current mode and submode
byte mode               = modeNone; // current mode
byte modeEprimeStreaming= 0;        // eprime submode, 0: not streaming, 1: bank 1, 2: bank 2
char modeExtendFunction = 'N';      // current function in extended mode (!='N' if one character read)
char modeExtendLed      = 'X';      // current led function in extended mode (X: none, I: input, O: output)
char modeExtendDetect   = 'X';      // current detect function in extended mode (X: none, S: sound, V: voice)

byte stateButtons = 0;        // state of buttons, one bit per button, 1 is pressed, 0 is not pressed
byte stateLeds = 0;           // state of LEDs, one bit per LED, 1 is on, 0 is off
unsigned long tButtons[8];    // (ms) time of buttonstate change (t0)

unsigned int voiceLow  = 0xffff; // lowest value during calibration - margin
unsigned int voiceHigh = 0;      // highest value during calibration + margin
unsigned int soundHigh = 0;      // highest sensors value during calibration + margin
unsigned int disconnectedSerial = 0;           // Counter disconnected usb

// epaper Display
// include library, include base class, make path known
#include "GxEPD.h"
// select the display class to use, only one
#include "GxGDEW0154Z04/GxGDEW0154Z04.cpp"  // 1.54" b/w/r
//#include GxEPD_BitmapExamples
// FreeFonts from Adafruit_GFX
#include <Fonts/FreeMonoBold9pt7b.h>
//#include <Fonts/FreeMonoBold12pt7b.h>
#include <Fonts/FreeMonoBold18pt7b.h>
//#include <Fonts/FreeMonoBold24pt7b.h>
#include "GxIO/GxIO_SPI/GxIO_SPI.cpp"
#include "GxIO/GxIO.cpp"

#define PIN_SPI_SS    (10)
#define PIN_SPI_MOSI  (11)
#define PIN_SPI_MISO  (12)
#define PIN_SPI_SCK   (13)
#define SS    (10)
#define MOSI  (11)
#define MISO  (12)
#define SCK   (13)
// initalize the  data ready and chip select pins:
//dr = 
//cs = 39;
//pinMode(dr, INPUT);
//pinMode(cs, OUTPUT);

// GxIO_SPI(SPIClass& spi, int8_t cs, int8_t dc, int8_t rst = -1, int8_t bl = -1);
GxIO_Class io(SPI, SS, 0, 1);
GxEPD_Class display(io, 1, 2);

void showInfoEpaper(String header = "BUTTONBOX", String type = "BITSI simple", String rate = "115200", String parity = "none", String dataBits = "8", String stopBits = "1"){
  //display.fillScreen(GxEPD_WHITE);
  display.setFont(&FreeMonoBold9pt7b);
  display.setCursor(0, 0);
  display.println();
  display.setTextColor(GxEPD_RED);
  display.println("Radboud University");
  display.setFont(&FreeMonoBold18pt7b);
  display.println("_________");
  display.setTextColor(GxEPD_BLACK);
  display.println(header);
  display.setFont(&FreeMonoBold9pt7b);
  display.println("type: " + type);
  display.println("rate: " + rate);
  display.println("parity: " + parity);
  display.println("data bits: " + dataBits);
  display.println("stop bits: " + stopBits);
  display.println("tsgdoc.socsci.ru.nl");
  display.update();
}

void modeEpaper(){
  if (mode == modeBitsi)
  {
    showInfoEpaper("BUTTONBOX", "BITSI simple", "115200", "none", "8", "1");
  }
  else if (mode == modeBitsiExtend)
  {
    showInfoEpaper("BUTTONBOX", "BITSI extend", "115200", "none", "8", "1");
  }
  else if (mode == modePersonal)
  {
    showInfoEpaper("PERSONAL", "documentation", "115200", "none", "8", "1");
  }
  else if (mode == modeEprime)
  {
    showInfoEpaper("BUTTONBOX", "Eprime", "19200", "none", "8", "1");
  }
}

void clearEpaper(){
  display.eraseDisplay();
}

// low level functions
void initDigitalInput(){
  for (int pinIndex = 0; pinIndex < 8; pinIndex++)
  {
    pinMode(pinInput[pinIndex], INPUT); // send 1
    digitalWrite(pinInput[pinIndex], HIGH);
  }
}

void initDigitalOutput(){
  for (int pinIndex = 0; pinIndex < 8; pinIndex++)
  {
    pinMode(pinOutput[pinIndex], OUTPUT); // send 1
    digitalWrite(pinOutput[pinIndex], LOW);
  }
}

void setOutput(byte b){
  byte pin = 0;
  for (byte mask = 00000001; mask > 0; mask <<= 1)
  { //iterate through bit mask
    if (b & mask)
    {                                     // if bitwise AND resolves to true
      digitalWrite(pinOutput[pin], HIGH); // send 1
    }
    else
    {                                    //if bitwise and resolves to false
      digitalWrite(pinOutput[pin], LOW); // send 0
    }
    pin++;
  }
}

// high level functions
void readButtons(byte n = 8){
  // Read the state of the first n buttons (n<=8)
  // Change button state only if dead time for that button is over.
  // Alter global variables: stateButtons, tButtons
  // used by handleButtonsEprime()
  ///
  boolean state;
  for (byte bit = 0; bit < n; bit++)
  {
    state = !digitalReadFast(pinInput[bit]); // true for pressed
    // check input state change, but not within debouncing interval
    if (bitRead(stateButtons, bit) != state && t0 - tButtons[bit] > timeDebounce)
    {
      // set state change
      bitWrite(stateButtons, bit, state);
      // make button insensitive during debouncing interval
      tButtons[bit] = t0;
    }
  }
}

void readButtonsNoDebounce(byte n = 8){
  // same as readButtons, but without detection of bouncing
  boolean state;
  for (byte bit = 0; bit < n; bit++)
  {
    state = !digitalRead(pinInput[bit]); // true for pressed
    if (bitRead(stateButtons, bit) != state)
    {
      bitWrite(stateButtons, bit, state);
      tButtons[bit] = t0;
    }
  }
}

void handleButtonsBitsi(){
  // Process input bits
  // Read the state of buttons and send characters A-Ha-h over serial connection in response
  // set global: stateButtons, tButtons
  // used by: bitsi and bitsiextend
  ///
  boolean state;
  for (byte bit = 0; bit < 8; bit++)
  {
    state = !digitalReadFast(pinInput[bit]); // true for pressed
    // check for input state change , but not within debouncing interval
    if (bitRead(stateButtons, bit) != state && t0 - tButtons[bit] > timeDebounce)
    {
      // send A-H for button down or a-h for button up
      if (state)
      {
        Serial.write('A' + bit);
        Serial.send_now();
        bitSet(stateButtons, bit);
      } else {
      Serial.write('a' + bit);
      Serial.send_now();
      bitClear(stateButtons, bit);
      }
    // make button insensitive during debouncing interval
    tButtons[bit] = t0;
    }

  //if(modeExtendLed=='I')
  //setLeds(stateButtons);
  }
}

void calibrateSound(){
  //takes 40ms
  soundHigh = 0; // sensor value
  analogRead(channelSound);
  analogRead(channelSound);
  setOutput(255);
  for (int i=0; i<5000; i++){
    unsigned int value = analogRead(channelSound);
    if (value > soundHigh)
      soundHigh = value;
  }
  soundHigh += marginSound;
  setOutput(0);
}

void calibrateVoice(){
  voiceLow = 0xffff;
  voiceHigh = 0;
  for (int i=0; i<5000; i++){
    unsigned int value = analogRead(channelVoice);
    if(value < voiceLow)
      voiceLow = value;
    if (value > voiceHigh)
      voiceHigh = value;
  } 
  voiceLow -= marginVoice;
  voiceHigh += marginVoice;
}

void sendMode(){
  readButtonsNoDebounce(); // read state of buttons to variable stateButtonss
  if (stateButtons == 0b00000011){ // A + B
    mode = modeBitsi;
    EEPROM.write(addressMode, mode);
  } else if (stateButtons == 0b00000101){ // A + C
    mode = modeBitsiExtend;
    EEPROM.write(addressMode, mode);
  }
  else if (stateButtons == 0b00001001)
  { // A + D
    mode = modePersonal;
    EEPROM.write(addressMode, mode);
  }
  else if (stateButtons == 0b00010001)
  { // A + E
    mode = modeEprime;
    EEPROM.write(addressMode, mode);
  }
  else
  {
    mode = EEPROM.read(addressMode);
  }
  if (mode == modeBitsiExtend || mode == modeEprime)
    calibrateSound();
  
  // leave mode as it is unless modeInitial is different from modeNone
  if (modeInitial != modeNone)
    mode = modeInitial;
  
  if (mode == modeBitsiExtend) {
    tone(pinSpeaker, 956,100);//laag
    delay(200);
    tone(pinSpeaker, 1915,100);//hoog
    delay(100);
    noTone(pinSpeaker);
  }

  if (mode == modeBitsi){
    Serial.begin(115200);
    Serial.print("BITSI mode, Ready!");
  } else if (mode == modeBitsiExtend){
    Serial.begin(115200);
    Serial.print("BITSI_extend mode, Ready!");
  }
  else if (mode == modePersonal)
  {
    Serial.begin(115200);
    Serial.print("Personal mode, Ready!");
  }
  else if (mode == modeEprime)
  {
    Serial.begin(19200);
    //Serial.write(0x7f);
  }
  if (mode != modeEprime){
    Serial.print(" w:");
    Serial.println(VERSION); // git version tag and hash
  }
  Serial.send_now();
  disconnectedSerial = millis() + 600;
}

void setup(){
  pinMode(13, OUTPUT); // send 1
  while (!Serial){
    digitalWrite(13, HIGH);
    delay(300);
    digitalWrite(13, LOW);
    delay(300);
    }
  SPI.begin();
  SPI.setBitOrder(MSBFIRST); 
  SPI.setClockDivider(SPI_CLOCK_DIV2);
  pinMode(pinAnalogIn1, INPUT); // pin written to to receive analog in from SPI
  pinMode(pinAnalogIn2, INPUT); // pin written to to receive analog in from SPI
  pinMode(pinAnalogIn3, INPUT); // pin written to to receive analog in from SPI
  pinMode(pinAnalogIn4, INPUT); // pin written to to receive analog in from SPI
  pinMode(pinAnalogOut1, OUTPUT);
  pinMode(pinAnalogOut2, OUTPUT);
  initDigitalInput();
  initDigitalOutput();
  setOutput(0);
  //setLeds(0);

 	// epaper
  //display.init();
 	//clearEpaper();

  // set mode if two buttons pressed
  sendMode();

  //writeAnalogA(0); // in first call after upload, value is ignored and will always cause 0V
  analogWrite(A21, 0);
  analogWrite(A22, 0);
  t0u = micros();
}

void loop(){
  t0 = millis();
  if (Serial){
    disconnectedSerial = millis() + 600; 
    //setOutput(255);
  //} else{
    //setOutput(0);
  }
  if (disconnectedSerial<millis()){
    //_reboot_Teensyduino_(); //does not run on windows, lost the serial com. On linux mac perfect
    //modeEpaper();
    CPU_RESTART;
  }

  if (mode == modeBitsi){
    if (Serial.available()){
      char c = Serial.read();
      //setLeds(c);
      setOutput(c);
    }
    handleButtonsBitsi();
  } else if (mode == modeBitsiExtend){
    while (Serial.available()){
      int i = Serial.read(); // int, -1 for no data available, this does not happen
      unsigned char c = (unsigned char) i;
      if (modeExtendFunction == 'N'){
        // c is first character of a two character bitsi extend command
        if (c == 'M' || c == 'Y' || c == 'Z' || c == 'C' || c == 'A' || c == 'D' || c == 'P' || c == 'X' || c == 'L') 
          modeExtendFunction = c;
        else if (c == 'T') // single character bitsi extend command
          tone(pinSpeaker, 1915, 100); // 1915 Hz, 100 ms square wave on internal speaker
      } else { 
        // modeExtendFunction is the second character of a two character bitsi extend command
        if(modeExtendFunction == 'D'){
          // start detection of sound or voice
          if (c == 'S') 
            modeExtendDetect = 'S';
          else if (c == 'V') 
            modeExtendDetect = 'V';
        } else if(modeExtendFunction == 'M'){
          // send marker
          setOutput(c);
          if(modeExtendLed == 'O')
            ;//setLeds(c);
        } else if(modeExtendFunction == 'P'){
          // like 'M' (send marker) but automatically stops after tPulse
          tPulseEnd = t0 + tPulse;
          setOutput(c);
          if(modeExtendLed == 'O')
            ;//setLeds(c);
        } else if(modeExtendFunction == 'Y'){
          // set value of analog channel 1, 0 - 4.3 V
          analogWrite(A21, i);
        } else if(modeExtendFunction == 'Z'){
          // set value of analog channel 2, 0 - 4.3 V
          analogWrite(A22, i);
        } else if(modeExtendFunction == 'X'){
          // set pulse duration
          tPulse = (unsigned long) c;
        } else if(modeExtendFunction == 'C'){
          if (c == 'S') {
            calibrateSound();
          } else if (c == 'V') 
            calibrateVoice();
        } else if(modeExtendFunction == 'A'){
          // read one of 4 analog channels (0-5 V) and send output as string
          if (c == '1'){
            Serial.println(analogRead(A14)); // "1234\r\n" (0-4096)
            Serial.send_now();}
          else if (c == '2'){
            Serial.println(analogRead(A15));
            Serial.send_now();}
          else if (c == '3') {
            Serial.println(analogRead(A16));
            Serial.send_now();}
          else if (c == '4') {
            Serial.println(analogRead(A17));
            Serial.send_now();}
        } else if(modeExtendFunction == 'L'){ // change LED, X: off, I: input, O: output
          if (c == 'X'){
            ;//setLeds(0);
            modeExtendLed = 'X';
          } else if (c == 'I'){
            modeExtendLed = 'I';
          } else if (c == 'O'){
            modeExtendLed = 'O';
          }
        }
        // reset to state before first character, even if second character was unrecognized
        modeExtendFunction = 'N'; 
      }
    } // end available bytes loop

    if (t0 >= tPulseEnd){
      // end pulse, started with 'P'
      tPulseEnd = 0xffffffff;
      setOutput(0);
      if(modeExtendLed == 'O')
        ;//setLeds(0);
      ;//setLeds(0);
    }
    handleButtonsBitsi(); // read buttons, set stateButtons, optionally set LEDs
    
    if (modeExtendDetect == 'S'){
      unsigned int value = analogRead(channelSound);
      if(value > soundHigh){
        Serial.print('S');
        Serial.send_now();
        modeExtendDetect = 'X';
      }
    } else if(modeExtendDetect == 'V'){
      unsigned int value = analogRead(channelVoice);
      if(value < voiceLow || value > voiceHigh){
        Serial.print('V');
        setOutput(255);
        delay(3);
        setOutput(0);
        Serial.send_now();
        modeExtendDetect = 'X';
      }
    }
  } else if (mode == modeEprime){ // end mode extend
    while (Serial.available()){
      byte c = byte(Serial.read()); // int, -1 for no data available (assume that this does not happen)
      // LEDs
      if(c & 0x40){ // lamp bank 1, 
        ;//setLeds((c & 0b00011111) | (stateLeds & 0b11100000)); // set LED A-E, keep LED F-H
        setOutput(c & 0b00011111);
      } else
        ;//setLeds(c << 5 | (stateLeds & 0b00011111)); // set LED F-H, keep LED A-E
      if(c & 0x80) // stream
        if(c & 0x20){ // key bank 1
          calibrateVoice();
          modeEprimeStreaming = 1; 
        } else
          modeEprimeStreaming = 2; 
      else
        modeEprimeStreaming = 0;
    } 
    unsigned tu = micros();
    if(modeEprimeStreaming){
      // wait to approximate requested streaming frequency (800Hz or 1600 Hz)
      if (tu-t0u < frametimeEprime)
        delayMicroseconds(frametimeEprime - tu + t0u);
      readButtons(); // set stateButtons
      // send byte
      if(modeEprimeStreaming==1){ // bank 1, lower 5 buttons
        uint8_t voice = 0;
        unsigned int value = analogRead(channelVoice);
        if(value < voiceLow || value > voiceHigh)
          voice = 32;
        Serial.print(char((stateButtons & 0b00011111) | voice));
        Serial.send_now();
      } else // bank 2, upper 3 buttons
        Serial.print(char(stateButtons >> 5)); 
        Serial.send_now();
      t0u = tu;
    }
    
  } // end mode ePrime
  //Serial.println(micros()-t0); 
}
